local ranges = {}

function ranges.range(start, stop, step)
    if not stop then
        stop = start
        start = 1
    end
    if not step then
        step = 1
    end
    local idx = start - step
    return function()
        idx = idx + step
        if step > 0 and idx > stop or step < 0 and idx < stop then
            return nil
        end
        return idx
    end
end

function ranges.wrap(f)
    return setmetatable({}, {
        __call = function(_, ...)
            return f(...)
        end,
        __bor = function(a, b)
            return b(a)
        end
    })
end

function ranges.filter(f)
    return ranges.wrap(function(t)
        if type(t) == "function" then
            return function()
                local k, v
                repeat
                    k, v = t()
                until not k or f(v or k)
                return k, v
            end
        end
        local idx = 0
        return function()
            local v
            repeat
                idx = idx + 1
                v = t[idx]
            until not v or f(v)
            return v
        end
    end)
end

function ranges.map(f)
    return ranges.wrap(function(t)
        if type(t) == "function" then
            return function()
                local k, v = t()
                if k and v then
                    return k, f(v)
                end
                if k then
                    return f(k)
                end
            end
        end
        local idx = 0
        return function()
            idx = idx + 1
            return t[idx] and f(t[idx])
        end
    end)
end

function ranges.slice(start, stop, step)
    return ranges.wrap(function(t)
        if not stop then
            stop = start
            start = 1
        end
        if not step then
            step = 1
        end
        if type(t) == "function" then
            local idx = 0
            return function()
                local k, v
                repeat
                    idx = idx + 1
                    k, v = t()
                until not k or idx >= start
                if not k or idx > stop then
                    return
                end
                return k, v
            end
        end
        local idx = 0
        return function()
            idx = idx + 1
            if idx < start or idx > stop or (idx - start) % step ~= 0 then
                return
            end
            return t[idx]
        end
    end)
end

function ranges.enumerate(t)
    if type(t) == "function" then
        local idx = 0
        return function()
            idx = idx + 1
            local v = t()
            if v == nil then
                return
            end
            return idx, v
        end
    end
    local idx = 0
    return function()
        idx = idx + 1
        if idx > #t then
            return
        end
        return idx, t[idx]
    end
end

function ranges.zip(...)
    local ts = { ... }
    return coroutine.wrap(function()
        local i, j = 1, 0
        repeat
            j = 1
            repeat
                coroutine.yield(ts[i][j])
                j = j + 1
            until j > #ts[i]
            i = i + 1
        until i > #ts
    end)
end

function ranges.take(n)
    return ranges.wrap(function(t)
        if type(t) == "function" then
            local idx = 0
            return function()
                idx = idx + 1
                if idx > n then
                    return
                end
                return t()
            end
        end
        local idx = 0
        return function()
            idx = idx + 1
            if idx > n then
                return
            end
            return t[idx]
        end
    end)
end

function ranges.drop(n)
    return ranges.wrap(function(t)
        if type(t) == "function" then
            local idx = 0
            return function()
                local k, v
                repeat
                    idx = idx + 1
                    k, v = t()
                until not k or idx > n
                return k, v
            end
        end
        local idx = 0
        return function()
            idx = idx + 1
            if idx <= n then
                return
            end
            return t[idx]
        end
    end)
end

function ranges.cycle()
    return ranges.wrap(function(t)
        if type(t) == "function" then
            local ts = {}
            local k, v
            repeat
                k, v = t()
                if k then
                    ts[#ts + 1] = v or k
                end
            until not k
            if #ts == 0 then
                return
            end
            return coroutine.wrap(function()
                while true do
                    for i = 1, #ts do
                        coroutine.yield(ts[i])
                    end
                end
            end)
        end
        return coroutine.wrap(function()
            while true do
                for i = 1, #t do
                    coroutine.yield(t[i])
                end
            end
        end)
    end)
end

function ranges.reverse()
    return ranges.wrap(function(t)
        if type(t) == "function" then
            local ts = {}
            local k, v
            repeat
                k, v = t()
                if k then
                    ts[#ts + 1] = v or k
                end
            until not k
            if #ts == 0 then
                return
            end
            return coroutine.wrap(function()
                for i = #ts, 1, -1 do
                    coroutine.yield(ts[i])
                end
            end)
        end
        return coroutine.wrap(function()
            for i = #t, 1, -1 do
                coroutine.yield(t[i])
            end
        end)
    end)
end

return ranges