local vector = require("util.geometry.vector")

local force = {
    forces = {},
}

function force.new(f)
    local self = {}

    for i, v in pairs(f) do
        self[i] = v
    end

    self.apply = self.apply or function(self, collider) end
    self.type = f.type or ""

    force.forces[#force.forces + 1] = self

    return self
end

function force.delete(f)
    for i = 1, #force.forces do
        if force.forces[i] == f then
            table.remove(force.forces, i)
            return
        end
    end
end

force.gravity = force.new({
    name = "gravity",
    type = "permanent",
    apply = function(_, collider)
        collider.force.y -= 9.81
    end,
})

force.drag = force.new({
    name = "drag",
    type = "permanent",
    density = {
        air = 1.2,
        water = 998,
    },
    velocity = {
        air = vector(0, 0, 0),
        water = vector(0, 0, 0),
    },
    apply = function(self, collider)
        --[[
            Fd = 0.5 * p * v^2 * A * Cd
        ]]
        local p = self.density[collider.fluid]
        local relative_velocity = collider.velocity - self.velocity[collider.fluid]
        local Cd = collider:drag_coeff()
        local drag = 0.5 * p * relative_velocity:length()^2 * collider:forward_area() * Cd
        collider.force += drag * -relative_velocity:normalize() * collider.inverse_mass
    end,
})

force.friction = force.new({
    name = "friction",
    type = "permanent",
    apply = function(_, collider)
        if collider.grounded then
            collider.force += -collider.velocity * collider.friction
        end
    end,
})

return force