local packet = {
    min_size = 4
}
setmetatable(packet, packet)

function packet:__call(context)
    local id = self:UnpackID(context:receive(4))
    local data, size = self:Fetch(context)
    return setmetatable({
        id = id,
        data = self:Deserialize(data),
        size = size
    }, self)
end

function packet:__tostring()
    return ("packet{id=%08X, size=%04X, data=%s}"):format(self.id, self.size, self.data)
end

function packet:Fetch(client)
    local size, data = 0, nil

    local response = client:receive(2)
    if not response then
        return nil
    end

    local size_ = response:sub(1, 2)

    if #size_ < 2 then
        error("invalid packet size")
    end

    for i = 1, 2 do
        size = (size << 8) | size_:byte(i)
    end

    response = client:receive(size)
    if not response then
        return nil
    end

    data = response:sub(1, size)

    return data, size
end

function packet:UnpackID(response)
    local id = 0

    local id_ = response:sub(1, 4)

    if #id_ < 4 then
        error("invalid packet ID")
    end

    for i = 1, 4 do
        id = (id << 8) | id_:byte(i)
    end

    return id
end

function packet:Pack(id, data, size)
    local id_ = ""
    for i = 1, 4 do
        id_ = string.char(id & 0xFF) .. id_
        id = id >> 8
    end

    if not data then
        return id_
    end

    if not size then
        return id_ .. data
    end

    if size > 0xFFFF then
        error("packet too large")
    end

    local size_ = ""
    for i = 1, 2 do
        size_ = string.char(size & 0xFF) .. size_
        size = size >> 8
    end

    return id_ .. size_ .. data
end

function packet:Serialize(data, lvl)
    if type(data) == "number" then
        return tostring(data), #tostring(data)
    elseif type(data) == "string" then
        return "\"" .. string.escape(data) .. "\"", #data + 2
    elseif type(data) == "table" then
        if type(data.Serialize) == "function" then
            return packet:Serialize(data:Serialize())
        else
            local str = "{"
            if table.isarray(data) then
                for i = 1, #data do
                    str = str .. packet:Serialize(data[i], true) .. ","
                end
            else
                for k, v in pairs(data) do
                    str = str .. "[" .. packet:Serialize(k, true) .. "]=" .. packet:Serialize(v, true) .. ","
                end
            end
            str = str:sub(1, #str - 1) .. "}"
            return str, #str
        end
    elseif type(data) == "nil" then
        return "nil", 3
    else
        error("Cannot serialize type " .. type(data))
    end
end

function packet:Deserialize(data, err)
    if not data then
        return nil, err
    end
    if type(data) == "string" then
        local f = load("return " .. data)
        if f then
            return f()
        else
            return data
        end
    else
        return data
    end
end

return packet