---@class Vector : Class
---@field x number
---@field y number
---@field z number?
---@field w number?
local Vector = {}

Vector.__index = function(self, key)
    do
        local v = rawget(self, key)
        if v then
            return v
        end
    end
    do
        local v = rawget(Vector, key)
        if v then
            return v
        end
    end
    if type(key) == "number" then
        if key == 1 then
            return self.x
        elseif key == 2 then
            return self.y
        elseif key == 3 then
            return self.z
        elseif key == 4 then
            return self.w
        end
    end
    for i = 1, #key do
        if key:sub(i, i) == "x" or key:sub(i, i) == "y" or key:sub(i, i) == "z" or key:sub(i, i) == "w" then
            continue
        end
        return nil
    end
    if #key == 2 then
        return Vector(self[key:sub(1, 1)], self[key:sub(2, 2)])
    elseif #key == 3 then
        return Vector(self[key:sub(1, 1)], self[key:sub(2, 2)], self[key:sub(3, 3)])
    elseif #key == 4 then
        return Vector(self[key:sub(1, 1)], self[key:sub(2, 2)], self[key:sub(3, 3)], self[key:sub(4, 4)])
    end
end
Vector.__newindex = function(self, key, value)
    if type(key) == "number" then
        if key == 1 then
            rawset(self, "x", value)
        elseif key == 2 then
            rawset(self, "y", value)
        elseif key == 3 then
            rawset(self, "z", value)
        elseif key == 4 then
            rawset(self, "w", value)
        else
            error("Invalid vector key: " .. key)
        end
        return
    end
    for i = 1, #key do
        if key:sub(i, i) == "x" or key:sub(i, i) == "y" or key:sub(i, i) == "z" or key:sub(i, i) == "w" then
            continue
        end
        rawset(self, key, value)
        return
    end
    local tval = type(value)
    if tval == "table" then
        if #key == 2 then
            rawset(self, key:sub(1, 1), rawget(value, "x"))
            rawset(self, key:sub(2, 2), rawget(value, "y"))
        elseif #key == 3 then
            rawset(self, key:sub(1, 1), rawget(value, "x"))
            rawset(self, key:sub(2, 2), rawget(value, "y"))
            rawset(self, key:sub(3, 3), rawget(value, "z"))
        elseif #key == 4 then
            rawset(self, key:sub(1, 1), rawget(value, "x"))
            rawset(self, key:sub(2, 2), rawget(value, "y"))
            rawset(self, key:sub(3, 3), rawget(value, "z"))
            rawset(self, key:sub(4, 4), rawget(value, "w"))
        end
    elseif tval == "number" then
        if #key == 2 then
            rawset(self, key:sub(1, 1), value)
            rawset(self, key:sub(2, 2), value)
        elseif #key == 3 then
            rawset(self, key:sub(1, 1), value)
            rawset(self, key:sub(2, 2), value)
            rawset(self, key:sub(3, 3), value)
        elseif #key == 4 then
            rawset(self, key:sub(1, 1), value)
            rawset(self, key:sub(2, 2), value)
            rawset(self, key:sub(3, 3), value)
            rawset(self, key:sub(4, 4), value)
        end
    end
end

function Vector.from(v)
    return Vector(v.x, v.y, v.z, v.w)
end

function Vector.ctor1(v)
    return Vector(v)
end

function Vector.ctor2(x, y)
    return Vector(x, y)
end

function Vector.ctor3(x, y, z)
    return Vector(x, y, z)
end

function Vector.ctor4(x, y, z, w)
    return Vector(x, y, z, w)
end

function Vector:__add(v)
    if type(self) == "number" then
        local tmp = v;v = self;self = tmp
    end
    local vt = type(v)
    if self.w then
        if vt == "number" then
            return Vector(self.x + v, self.y + v, self.z + v, self.w + v)
        elseif vt == "table" then
            return Vector(self.x + v.x, self.y + v.y, self.z + v.z, self.w + v.w)
        end
    elseif self.z then
        if vt == "number" then
            return Vector(self.x + v, self.y + v, self.z + v)
        elseif vt == "table" then
            return Vector(self.x + v.x, self.y + v.y, self.z + v.z)
        end
    else
        if vt == "number" then
            return Vector(self.x + v, self.y + v)
        elseif vt == "table" then
            return Vector(self.x + v.x, self.y + v.y)
        end
    end
    error("Invalid vector type")
end

function Vector:__sub(v)
    if type(self) == "number" then
        local tmp = v;v = self;self = tmp
    end
    local vt = type(v)
    if self.w then
        if vt == "number" then
            return Vector(self.x - v, self.y - v, self.z - v, self.w - v)
        elseif vt == "table" then
            return Vector(self.x - v.x, self.y - v.y, self.z - v.z, self.w - v.w)
        end
    elseif self.z then
        if vt == "number" then
            return Vector(self.x - v, self.y - v, self.z - v)
        elseif vt == "table" then
            return Vector(self.x - v.x, self.y - v.y, self.z - v.z)
        end
    else
        if vt == "number" then
            return Vector(self.x - v, self.y - v)
        elseif vt == "table" then
            return Vector(self.x - v.x, self.y - v.y)
        end
    end
    error("Invalid vector type")
end

function Vector:__mul(v)
    if type(self) == "number" then
        local tmp = v;v = self;self = tmp
    end
    local vt = type(v)
    if self.w then
        if vt == "number" then
            return Vector(self.x * v, self.y * v, self.z * v, self.w * v)
        elseif vt == "table" then
            return Vector(self.x * v.x, self.y * v.y, self.z * v.z, self.w * v.w)
        end
    elseif self.z then
        if vt == "number" then
            return Vector(self.x * v, self.y * v, self.z * v)
        elseif vt == "table" then
            return Vector(self.x * v.x, self.y * v.y, self.z * v.z)
        end
    else
        if vt == "number" then
            return Vector(self.x * v, self.y * v)
        elseif vt == "table" then
            return Vector(self.x * v.x, self.y * v.y)
        end
    end
    error("Invalid vector type")
end

function Vector:__div(v)
    if type(self) == "number" then
        local tmp = v;v = self;self = tmp
    end
    local vt = type(v)
    if self.w then
        if vt == "number" then
            return Vector(self.x / v, self.y / v, self.z / v, self.w / v)
        elseif vt == "table" then
            return Vector(self.x / v.x, self.y / v.y, self.z / v.z, self.w / v.w)
        end
    elseif self.z then
        if vt == "number" then
            return Vector(self.x / v, self.y / v, self.z / v)
        elseif vt == "table" then
            return Vector(self.x / v.x, self.y / v.y, self.z / v.z)
        end
    else
        if vt == "number" then
            return Vector(self.x / v, self.y / v)
        elseif vt == "table" then
            return Vector(self.x / v.x, self.y / v.y)
        end
    end
    error("Invalid vector type")
end

function Vector:__unm()
    if self.w then
        return Vector(-self.x, -self.y, -self.z, -self.w)
    elseif self.z then
        return Vector(-self.x, -self.y, -self.z)
    else
        return Vector(-self.x, -self.y)
    end
    error("Invalid vector type")
end

function Vector:__eq(v)
    return self.x == v.x and self.y == v.y and self.z == v.z and self.w == v.w
end

function Vector:__tostring()
    if self.w then
        return string.format("(%f, %f, %f, %f)", self.x, self.y, self.z, self.w)
    elseif self.z then
        return string.format("(%f, %f, %f)", self.x, self.y, self.z)
    else
        return string.format("(%f, %f)", self.x, self.y)
    end
    error("Invalid vector type")
end

function Vector:__len()
    return math.sqrt(self.x * self.x + self.y * self.y + (self.z or 0) * (self.z or 0) + (self.w or 0) * (self.w or 0))
end

function Vector:normalize()
    return self / #self
end

function Vector:dot(v)
    return self.x * v.x + self.y * v.y + (self.z or 0) * (v.z or 0) + (self.w or 0) * (v.w or 0)
end

function Vector:cross(v)
    return Vector(
        self.y * v.z - self.z * v.y,
        self.z * v.x - self.x * v.z,
        self.x * v.y - self.y * v.x
    )
end

function Vector:distance(v)
    return math.sqrt((self.x - v.x) ^ 2 + (self.y - v.y) ^ 2 + ((self.z or 0) - (v.z or 0)) ^ 2 + ((self.w or 0) - (v.w or 0)) ^ 2)
end

function Vector:length()
    return #self
end

function Vector:length_squared()
    return self.x * self.x + self.y * self.y + (self.z or 0) * (self.z or 0) + (self.w or 0) * (self.w or 0)
end

function Vector:angle()
    return math.atan2(self.y, self.x)
end

function Vector:angle_between(v)
    return math.acos(self:dot(v) / (#self * #v))
end

function Vector:rotate(angle)
    local s = math.sin(angle)
    local c = math.cos(angle)
    return Vector(self.x * c - self.y * s, self.x * s + self.y * c)
end

function Vector:rotate_around(angle, origin)
    return (self - origin):rotate(angle) + origin
end

function Vector:rotate_around_axis(angle, axis)
    local s = math.sin(angle)
    local c = math.cos(angle)
    local v = axis:normalize()
    local res = Vector(0, 0, 0)
    res.x = (c + (1 - c) * v.x * v.x) * self.x + ((1 - c) * v.x * v.y - v.z * s) * self.y + ((1 - c) * v.x * v.z + v.y * s) * self.z
    res.y = ((1 - c) * v.x * v.y + v.z * s) * self.x + (c + (1 - c) * v.y * v.y) * self.y + ((1 - c) * v.y * v.z - v.x * s) * self.z
    res.z = ((1 - c) * v.x * v.z - v.y * s) * self.x + ((1 - c) * v.y * v.z + v.x * s) * self.y + (c + (1 - c) * v.z * v.z) * self.z
    return res
end

function Vector:project(v)
    return v * (self:dot(v) / v:length_squared())
end

setmetatable(Vector, {
    __call = function(self, x, y, z, w)
        local v = setmetatable({
            x = x or 0,
            y = y or 0,
            z = z,
            w = w
        }, Vector)
        return v
    end
})

return Vector