local Vector = require("geometry.vector")

---@class Rectangle : Class
local Rectangle = import("class", "local")()

---@param self Rectangle
function Rectangle:__init(x, y, w, h, b)
    if type(x) == "table" then
        self.x = x.x
        self.y = x.y
        self.w = x.w or x.width
        self.h = x.h or x.height
        self.b = x.b or x.border
    else
        self.x = x
        self.y = y
        self.w = w
        self.h = h
        self.b = b
    end
end

function Rectangle:__tostring()
    return string.format("Rectangle(%s, %s, %s, %s, %s)", self.x, self.y, self.w, self.h, self.b)
end

function Rectangle:__add(other)
    if type(other) == "number" then
        return Rectangle(self.x + other, self.y + other, self.w + other, self.h + other, (self.b or 0) + other)
    else
        return Rectangle(self.x + other.x, self.y + other.y, self.w + other.w, self.h + other.h, (self.b or 0) + (other.b or 0))
    end
end

function Rectangle:clone(x, y, w, h, b)
    return Rectangle(self.x + (x or 0), self.y + (y or 0), self.w + (w or 0), self.h + (h or 0), (self.b or b) and ((self.b or 0) + (b or 0)) or nil)
end

---requires gui import
local gui
function Rectangle:draw(color, rounded)
    gui = gui or import("gui")
    if self.b then
        gui.stroke_weight(self.b)
        gui.stroke(color)
        gui.no_fill()
        gui.rectangle(
            self.x - self.b * .5, self.y - self.b * .5,
            self.w + self.b, self.h + self.b,
            color)
    else
        gui.no_stroke()
        gui.fill(color)
        gui.rectangle(
            self.x, self.y,
            self.w, self.h,
            color)
    end
end

--[[
    (self.x - self.b, self.y - self.b)-----------------------------(self.x + self.w + self.b, self.y - self.b)
    |   (self.x, self.y)-------------------------------------------------(self.x + self.w, self.y)           |
    |       |                                                                       |                        |
    |       |                                                                       |                        |
    |       |                                                                       |                        |
    |   (self.x, self.y + self.h)----------------------------------------(self.x + self.w, self.y + self.h)  |
    (self.x - self.b, self.y + self.h + self.b)-----------(self.x + self.w + self.b, self.y + self.h + self.b)
]]

function Rectangle:contains(v2, inner)
    if self.b and not inner then
        return not (v2.x > self.x and v2.x < self.x + self.w
            and v2.y > self.y and v2.y < self.y + self.h) -- not inside inner
            and (v2.x > self.x - self.b and v2.x < self.x + self.w + self.b
            and v2.y > self.y - self.b and v2.y < self.y + self.h + self.b) -- inside outer
    else
        return v2.x > self.x and v2.x < self.x + self.w
           and v2.y > self.y and v2.y < self.y + self.h
    end
end

function Rectangle:right_border_contains(v2, b)
    return v2.x > self.x + self.w and v2.x < self.x + self.w + (b or self.b)
       and v2.y > self.y - (b or self.b) and v2.y < self.y + self.h + (b or self.b)
end

function Rectangle:left_border_contains(v2, b)
    return v2.x > self.x - (b or self.b) and v2.x < self.x
       and v2.y > self.y - (b or self.b) and v2.y < self.y + self.h + (b or self.b)
end

function Rectangle:top_border_contains(v2, b)
    return v2.x > self.x - (b or self.b) and v2.x < self.x + self.w + (b or self.b)
       and v2.y > self.y - (b or self.b) and v2.y < self.y
end

function Rectangle:top_right_contains(v2, b)
    return v2.x > self.x + self.w - (b or self.b) and v2.x < self.x + self.w
       and v2.y > self.y - (b or self.b) and v2.y < self.y
end

function Rectangle:bottom_border_contains(v2, b)
    return v2.x > self.x - (b or self.b) and v2.x < self.x + self.w + (b or self.b)
       and v2.y > self.y + self.h and v2.y < self.y + self.h + (b or self.b)
end

function Rectangle:bottom_right_contains(v2, b)
    return v2.x > self.x + self.w - (b or self.b) and v2.x < self.x + self.w + (b or self.b)
       and v2.y > self.y + self.h - (b or self.b) and v2.y < self.y + self.h + (b or self.b)
end

function Rectangle:bottom_left_contains(v2, b)
    return v2.x > self.x - (b or self.b) and v2.x < self.x
       and v2.y > self.y + self.h and v2.y < self.y + self.h + (b or self.b)
end

function Rectangle:bottom_left()
    return Vector(self.x - (self.b or 0), self.y + self.h + (self.b or 0))
end

function Rectangle:bottom_right()
    return Vector(self.x + self.w + (self.b or 0), self.y + self.h + (self.b or 0))
end

function Rectangle:topLeft()
    return Vector(self.x - (self.b or 0), self.y - (self.b or 0))
end

function Rectangle:topRight()
    return Vector(self.x + self.w + (self.b or 0), self.y - (self.b or 0))
end

function Rectangle:unpack()
    return self.x, self.y, self.w, self.h, self.b
end

function Rectangle:intersect(other)
    return self.x < other.x + other.w and self.x + self.w > other.x
       and self.y < other.y + other.h and self.y + self.h > other.y
end

function Rectangle:inside(other)
    return self.x > other.x and self.x + self.w < other.x + other.w
       and self.y > other.y and self.y + self.h < other.y + other.h
end

---@return Vector
function Rectangle:distance(other)
    local dx = math.max(other.x - (self.x + self.w), self.x - (other.x + other.w), 0)
    local dy = math.max(other.y - (self.y + self.h), self.y - (other.y + other.h), 0)
    return Vector(dx, dy)
end

---@return Rectangle
return function (...)
    return Rectangle(...)
end