local class_storage = require("ecs.entity_storage")
local entities = {}
local entity_count = 0
local class_boxes = {}
local component_boxes = {}
local entity_component_map = {}
local entity_class_map = {}

local function insert(entity)
    if #entities < entity_count then
        local id = #entities + 1
        entities[id] = entity
        return id
    end
    entity_count = entity_count + 1
    entities[entity_count] = entity
    return entity_count
end

local function remove(entity)
    entities[entity.__id] = nil
    if entity.__id == entity_count then
        for i = entity_count, 1, -1 do
            if entities[i] then
                entity_count = i
                break
            end
        end
    end
end

local function new(class, args)
    if not class_storage[class] then
        error("Class " .. class .. " does not exist")
    end
    local e, classes, components = class_storage[class].class.new(args)
    e.__id = insert(e)
    for i, c in pairs(classes) do
        if not class_boxes[c] then
            class_boxes[c] = {}
        end
        class_boxes[c][e.__id] = true
    end
    entity_component_map[e.__id] = {}
    for i, c in pairs(components) do
        if not component_boxes[c] then
            component_boxes[c] = {}
        end
        component_boxes[c][e.__id] = true
    end
    entity_class_map[e.__id] = classes
    return e
end

local function delete(entity)
    if not entities[entity.__id] then
        return false
    end
    for i, c in pairs(entity_class_map[entity.__id]) do
        class_boxes[c][entity.__id] = nil
    end
    for i, c in pairs(entity_component_map[entity.__id]) do
        component_boxes[c][entity.__id] = nil
    end
    remove(entity)
    return true
end

local function for_each(func, args)
    for i = 1, entity_count do
        if entities[i] then
            func(entities[i], args)
        end
    end
end

local function get(id)
    return entities[id]
end

local function count()
    return entity_count
end

local function find_by_class(class)
    if class_boxes[class] then
        return get(next(class_boxes[class]))
    end
    return nil
end

local function find_all_by_class(class)
    if class_boxes[class] then
        return class_boxes[class]
    end
    return {}
end

local function find_by_component(component)
    if component_boxes[component] then
        return get(next(component_boxes[component]))
    end
    return nil
end

local function find_all_by_component(component)
    if component_boxes[component] then
        return component_boxes[component]
    end
    return {}
end

local function add_component(entity, component)
    if not entity_component_map[entity.__id] then
        entity_component_map[entity.__id] = {}
    end
    entity_component_map[entity.__id][component] = true
    entity = class_storage[entity_class_map[entity.__id][#entity_class_map[entity.__id]]].class.add_component(entity, component)
    return entity
end

local function remove_component(entity, component)
    if not entity_component_map[entity.__id] then
        return
    end
    entity_component_map[entity.__id][component] = nil
    class_storage[entity_class_map[entity.__id][#entity_class_map[entity.__id]]].class.remove_component(entity, component)
end

local function has_component(entity, component)
    return entity_component_map[entity.__id][component]
end

local function get_components(entity)
    return entity_component_map[entity.__id]
end

local function get_class(entity)
    return entity_class_map[entity.__id][#entity_class_map[entity.__id]]
end

local function get_classes(entity)
    return entity_class_map[entity.__id]
end

local function is(entity, class)
    local map = entity_class_map[entity.__id]
    for i = 1, #map do
        if map[i] == class then
            return true
        end
    end
    return false
end

return {
    new = new,
    delete = delete,
    for_each = for_each,
    entity = require("ecs.entity"),
    component = require("ecs.component"),
    remove = remove,
    get = get,
    entities = function()
        return entities
    end,
    count = count,
    find_by_class = find_by_class,
    find_all_by_class = find_all_by_class,
    find_by_component = find_by_component,
    find_all_by_component = find_all_by_component,
    add_component = add_component,
    remove_component = remove_component,
    has_component = has_component,
    get_components = get_components,
    get_class = get_class,
    get_classes = get_classes,
    is = is
}