---@class Coroutine : Class
---@field status string
---@field yieldable boolean
---@field running boolean
---@field f thread
local Coroutine = import("class", "local")()

function Coroutine:__init(f, s)
    self.f_self = s
    self.f_store = f
    local fenv = getfenv(f)
    function fenv.yield(...)
        coroutine.yield(...)
    end
    function fenv.yield_if(cond, ...)
        if type(cond) == "function" then
            if cond(table.unpack({...})) then
                coroutine.yield()
            end
        elseif cond then
            coroutine.yield(...)
        end
    end
    function fenv.yield_if_not(cond, ...)
        if type(cond) == "function" then
            if not cond(table.unpack({...})) then
                coroutine.yield()
            end
        elseif not cond then
            coroutine.yield(...)
        end
    end
    function fenv.await(cond, ...)
        while (not cond(table.unpack({...}))) do
            coroutine.yield()
        end
    end
    function fenv.time_passed(t, f, args)
        local time_store = os.clock() + t
        return function()
            if f then f(table.unpack(args or {})) end
            return os.clock() > time_store
        end
    end
    self.f = coroutine.create(f)
end

---@param f fun(co:Coroutine, ...):any
function Coroutine.new(f, s)
    return Coroutine(f, s)
end

import("extensions.debug", "local")

function Coroutine:__call(...)
    if self.status == "normal" then return end
    local status, err;
    if self.f_self then
        status, err = coroutine.resume(self.f, self.f_self, self, ...)
    else
        status, err = coroutine.resume(self.f, self, ...)
    end
    if not status then
        printf("Error in coroutine:\n" .. err .. "\nfunction:\n\t" .. debug.functrace(self.f_store) .. debug.traceback(self.f))
    end
    if self.status == "dead" then
        self.f = coroutine.create(self.f_store)
    end
    return err
end

---@param self Coroutine
function Coroutine.__properties.yieldable:get()
    return coroutine.isyieldable(self.f)
end

function Coroutine.__properties.running:get()
    return (coroutine.running())
end

---@param self Coroutine
function Coroutine.__properties.status:get()
    return coroutine.status(self.f)
end

---@return Coroutine
return function(f, s)
    return Coroutine.new(f, s)
end